/*

FractalDrawer - A Fractal Drawer Applet/Application
Copyright (C) 2001  Laurentiu Cristofor


This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
USA

*/

/*
 * Fractal.java       v1.0   11/09/2001
 *
 * Copyright (c) 2001 Laurentiu Cristofor. All Rights Reserved.
 *
 */

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.geom.Point2D;

public abstract class Fractal
{
  // constants for indicating the symmetry of the fractal
  protected static final int NO_SYMMETRY           = 0;
  protected static final int HORIZONTAL_SYMMETRY   = 1;
  protected static final int VERTICAL_SYMMETRY     = 2;
  protected static final int HV_SYMMETRY           = 3;
  protected static final int CENTRAL_SYMMETRY      = 4;

  // constants for indicating how the fractal view is centered
  protected static final int NOT_CENTERED          = 0;
  protected static final int HORIZONTALLY_CENTERED = 1;
  protected static final int VERTICALLY_CENTERED   = 2;
  protected static final int HV_CENTERED           = 3;

  // if the series gets farther from the origin than this we consider
  // that the point does not belong to the fractal set
  protected static final double DISTANCE_TO_ORIGIN_SQUARED = 4.0;

  // how the fractal view is centered, and its type of symmetry
  protected int TYPE_CENTERED, TYPE_SYMMETRY;

  // the height and width of the display area in pixels
  // this the area where we can draw
  protected int display_height, display_width;

  // the height and width of the used display area in pixels
  // this the area where we actually draw
  protected int used_display_height, used_display_width;

  // these are the fractal space boundaries 
  protected double x_min, y_min, x_max, y_max;

  // after this number of iterations we consider that a point belongs
  // to the fractal set
  protected int iteration_threshold;

  // the color palette used to display the fractal
  protected Color palette[];

  // the step we use in moving through the fractal space
  protected double delta;

  // the graphical representation of the fractal set, each entry
  // correponding to a display pixel, and representing the Color
  // assigned to that pixel (black if it belongs to the fractal set or
  // a color based on the number of iterations needed to see that it
  // does not belong to the fractal set)
  protected Color[][] fractal; 

  protected static final double ALMOST_ZERO = 0.0001;

  // this method checks whether the parameters are symmetric with
  // respect to 0
  protected static boolean areSymmetric(double min, double max)
  {
    if (min >= 0 || max <= 0)
      return false;

    if (Math.abs((min + max) / (max - min)) < ALMOST_ZERO)
      return true;
    else 
      return false;
  }

  /**
   * Initializes the fractal.
   *
   * @param display_width   the width of the display 
   * @param display_height   the height of the display
   * @param x_min   lowest real value of fractal space that we explore
   * @param y_min   lowest imaginary value of fractal space
   * @param x_max   largest real value of fractal space
   * @param y_max   largest imaginary value of fractal space
   * @param iteration_threshold   how many iterations we perform until 
   * we accept a point (unless we reject it after fewer iterations)
   * @param palette   an array of colors used to draw the fractal
   *
   * @exception InvalidArgumentException thrown if display width or
   * height are negative, or if x_min > x_max, or if y_min > y_max, or
   * if palette is null, or if iteration_threshold is <= 0, or if
   * palette has size less than iteration_threshold.
   **/
  public Fractal(int display_width, int display_height,
		 double x_min, double y_min, double x_max, double y_max,
		 int iteration_threshold,
		 Color[] palette)
  {
    if (display_width <= 0 || display_height <= 0
	|| x_max <= x_min || y_max <= y_min 
	|| palette == null
	|| iteration_threshold <= 0 || iteration_threshold > palette.length)
      throw new IllegalArgumentException();

    TYPE_CENTERED = NOT_CENTERED;
    TYPE_SYMMETRY = NO_SYMMETRY;

    this.display_width = display_width;
    this.display_height = display_height;
    this.x_min = x_min;
    this.y_min = y_min;
    this.x_max = x_max;
    this.y_max = y_max;
    this.iteration_threshold = iteration_threshold;
    this.palette = palette;

    setupDisplaySettings();
  }

  // this method determines if the fractal space is centered,
  // determines how to fit the fractal in the display, and computes the
  // value of delta which will be used in traversing the fractal space
  protected void setupDisplaySettings()
  {
    if (areSymmetric(x_min, x_max))
      if (areSymmetric(y_min, y_max))
	TYPE_CENTERED = HV_CENTERED;
      else
	TYPE_CENTERED = VERTICALLY_CENTERED;
    else
      if (areSymmetric(y_min, y_max))
	TYPE_CENTERED = HORIZONTALLY_CENTERED;
      else
	TYPE_CENTERED = NOT_CENTERED;

    double display_ratio = (double)display_width/(double)display_height;
    double view_ratio = Math.abs((x_max - x_min)/(y_max - y_min));

    // if view is wider it won't use the entire display height
    if (view_ratio > display_ratio)
      {
	delta = Math.abs(x_max - x_min) / display_width;
	used_display_height = (int)(display_width / view_ratio);
	used_display_width = display_width;
      }
    // if view is taller it won't use the entire display width
    else
      {
	delta = Math.abs(y_max - y_min) / display_height;
	used_display_width = (int)(display_height * view_ratio);
	used_display_height = display_height;
      }
  }

  private static void putPixel(Graphics g, int x, int y, Color c)
  {
    g.setColor(c);
    g.drawLine(x, y, x, y);
  }

  /**
   * Displays the current view of the fractal on a given graphics context.
   *
   * @param g   the graphics context
   **/
  public void display(Graphics g)
  {
    if (fractal == null)
      build();

    for (int i = 0; i < used_display_width; i++)
      for (int j = 0; j < used_display_height; j++)
	putPixel(g, i, j, fractal[i][j]);
  }

  /**
   * Converts a display coordinate to a fractal space coordinate. The
   * display coordinate is "clipped" to the used display boundaries
   * before conversion.
   *
   * @param p   a display coordinate
   * @return   the corresponding fractal space coordinate
   **/
  public Point2D.Double convertToFractalSpace(Point p)
  {
    // "clip" point
    if (p.x < 0)
      p.x = 0;
    if (p.x > used_display_width)
      p.x = used_display_width - 1;
    if (p.y < 0)
      p.y = 0;
    if (p.y > used_display_height)
      p.y = used_display_height - 1;

    double x, y;

    x = x_min + delta * p.x;
    y = y_max - delta * p.y;

    return new Point2D.Double(x, y);
  }

  /**
   * Enhances the fractal if the new threshold is higher, has opposite
   * effect otherwise.
   *
   * @param new_threshold   the new threshold
   * @param new_palette   the new palette to be used with this threshold
   * 
   * @exception IllegalArgumentException new_threshold is <= 0 or
   * palette is null or has size less than new_threshold
   **/
  public void enhance(int new_threshold, Color[] new_palette)
  {
    if (palette == null
	|| iteration_threshold <= 0 || iteration_threshold > palette.length)
      throw new IllegalArgumentException();

    iteration_threshold = new_threshold;
    palette = new_palette;
    fractal = null; // force rebuild in display()
  }

  /**
   * Zooms in the fractal space; the points received represent the
   * opposite corners of a region of the display into which we want to
   * zoom, this region will have to be converted to a region in the
   * fractal space. The region is clipped to the used display
   * boundaries before that conversion.
   *
   * @param x1   x coordinate of first point
   * @param y1   y coordinate of first point
   * @param x2   x coordinate of second point
   * @param y2   y coordinate of second point
   **/
  public void zoom(int x1, int y1, 
		   int x2, int y2)
  {
    // clip region
    if (x1 < 0)
      x1 = 0;
    if (x2 < 0)
      x2 = 0;
    if (x1 >= used_display_width)
      x1 = used_display_width - 1;
    if (x2 >= used_display_width)
      x2 = used_display_width - 1;

    if (y1 < 0)
      y1 = 0;
    if (y2 < 0)
      y2 = 0;
    if (y1 >= used_display_height)
      y1 = used_display_height - 1;
    if (y2 >= used_display_height)
      y2 = used_display_height - 1;
    
    // can't zoom in empty region
    if (x1 == x2 || y1 == y2)
      return;

    // arrange for point 1 to be upper-left point of region
    // and for point 2 to be the bottom-right point
    int tmp;
    if (x2 < x1)
      {
	tmp = x1;
	x1 = x2;
	x2 = tmp;
      }
    if (y2 < y1)
      {
	tmp = y1;
	y1 = y2;
	y2 = tmp;
      }

    // now compute the region in the fractal space
    x_min += delta * x1;
    y_min += delta * (used_display_height - y2);
    x_max -= delta * (used_display_width - x2);
    y_max -= delta * y1;

    zoom(x_min, y_min, x_max, y_max);
  }

  /**
   * Zooms in the fractal space by setting new boundaries of this
   * space.
   *
   * @param x_min   lowest real value of fractal space that we explore
   * @param y_min   lowest imaginary value of fractal space
   * @param x_max   largest real value of fractal space
   * @param y_max   largest imaginary value of fractal space
   *
   * @exception InvalidArgumentException thrown if x_min > x_max or if
   * y_min > y_max
   **/
  public void zoom(double x_min, double y_min, 
		   double x_max, double y_max)
  {
    if (x_max <= x_min || y_max <= y_min)
      throw new IllegalArgumentException();

    this.x_min = x_min;
    this.y_min = y_min;
    this.x_max = x_max;
    this.y_max = y_max;

    setupDisplaySettings();
    fractal = null; // force rebuild in display()
  }

  /**
   * Resizes the display by specifying its new dimensions.
   *
   * @param display_width   the width of the display 
   * @param display_height   the height of the display
   *
   * @exception InvalidArgumentException thrown if display width or
   * height are negative
   **/
  public void resizeDisplay(int display_width, int display_height)
  {
    if (display_width <= 0 || display_height <= 0)
      throw new IllegalArgumentException();

    this.display_width = display_width;
    this.display_height = display_height;

    setupDisplaySettings();
    fractal = null; // force rebuild in display()
  }

  /**
   * This method should be overriden by subclasses to compute whether
   * a point belongs to the fractal space.
   *
   * @param x   the real value of the point
   * @param y   the complex value of the point
   **/
  public abstract int computeFractalValue(double x, double y);

  private boolean canUseVerticalSymmetry()
  {
    return ((TYPE_SYMMETRY == VERTICAL_SYMMETRY 
	     || TYPE_SYMMETRY == HV_SYMMETRY)
	    && (TYPE_CENTERED == VERTICALLY_CENTERED
		|| TYPE_CENTERED == HV_CENTERED));
  }

  private boolean canUseHorizontalSymmetry()
  {
    return ((TYPE_SYMMETRY == HORIZONTAL_SYMMETRY 
	     || TYPE_SYMMETRY == HV_SYMMETRY)
	    && (TYPE_CENTERED == HORIZONTALLY_CENTERED 
		|| TYPE_CENTERED == HV_CENTERED));
  }

  private boolean canUseCentralSymmetry()
  {
    return (TYPE_SYMMETRY == CENTRAL_SYMMETRY
	    && TYPE_CENTERED == HV_CENTERED);
  }

  // this method builds the fractal in memory and uses whatever
  // symmetry information we have to speed up the computations
  protected void build()
  {
    int i, j, iteration;
    int i_max = used_display_width, j_max = used_display_height;
    double x, y;
    Color c;

    // free some space before allocating a new array
    System.gc();

    fractal = new Color[used_display_width][used_display_height];

    if (canUseVerticalSymmetry())
      i_max = used_display_width / 2;

    if (canUseHorizontalSymmetry())
      j_max = used_display_height / 2;

    if (canUseCentralSymmetry())
      j_max = used_display_height / 2;

    for (i = 0, x = x_min; i < i_max; i++, x += delta)
      for (j = 0, y = y_max; j < j_max; j++, y -= delta)
	{
	  iteration = computeFractalValue(x, y);

	  // determine the color of the point based on the number of
	  // iterations performed
	  c = (iteration < iteration_threshold) 
	    ? palette[iteration] : Color.black;

	  fractal[i][j] = c;

	  // see if we can draw a symmetric point

	  if (canUseVerticalSymmetry())
	    fractal[used_display_width - i - 1][j] = c;
	  
	  if (canUseHorizontalSymmetry())
	    fractal[i][used_display_height - j - 1] = c;

	  if (canUseCentralSymmetry())
	    fractal[used_display_width - i - 1][used_display_height - j - 1] 
	      = c;
	}
  }
}
