/*

FractalDrawer - A Fractal Drawer Applet/Application
Copyright (C) 2001  Laurentiu Cristofor


This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
USA

*/

/*
 * FractalDrawer.java       v1.1   12/07/2001
 *
 * Copyright (c) 2001 Laurentiu Cristofor. All Rights Reserved.
 *
 */

import java.applet.*;
import java.awt.event.*;
import java.awt.*;
import java.awt.geom.Point2D;
import javax.swing.*;

public class FractalDrawer extends JApplet
{
  public static final String APPNAME   = "Fractal Drawer";
  public static final String VERSION   = "v1.1 (Dec. 2001)";
  public static final String COPYRIGHT = "(C)2001 Laurentiu Cristofor";
  public static final String INFO      = APPNAME + " " + VERSION 
    + "   " + COPYRIGHT;

  private FractalDrawPanel draw_panel;
  private FractalSettings fractal_settings;

  // indicates if we execute the program as an application
  static boolean isApplication;

  public String getAppletInfo() 
  {
    return INFO 
      + "\nA fractal drawing program distributed under GNU GPL." 
      + "\nSee file gpl.txt for license information.";
  }

  public void init() 
  {
    draw_panel = new FractalDrawPanel();
    getContentPane().add("Center", draw_panel);

    fractal_settings = new FractalSettings(this, draw_panel);
    getContentPane().add("South", fractal_settings);
  }

  public void destroy() 
  {
    getContentPane().remove(draw_panel);
    getContentPane().remove(fractal_settings);
  }

  public static void main(String args[]) 
  {
    isApplication = true;

    JFrame main_frame = new JFrame(APPNAME);

    FractalDrawer fractalDrawer = new FractalDrawer();
    fractalDrawer.init();
    fractalDrawer.start();

    main_frame.getContentPane().add("Center", fractalDrawer);
    main_frame.setSize(400, 400);
    main_frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    main_frame.show();
  }
}

class FractalDrawPanel 
  extends JPanel 
  implements MouseMotionListener, MouseListener
{
  // default values for palette construction
  private static final int DEF_PAL_HIGH = 246;
  private static final int DEF_PAL_LOW  = 98;
  private static final int DEF_PAL_DEC  = 32;

  private FractalSettings settings;

  private Fractal fractal;
  private Color[] palette;

  private int old_width, old_height;
  private int x1, y1, x2, y2;

  private boolean bDragging;

  // this general method generates a palette by first modifying the
  // red, then the green, then the blue, and then repeating this cycle
  private void constructPalette(int size, int high, int low, int decrement)
  {
    if (high < low || low < 0 || high > 255
	|| decrement < 0 || decrement > high - low || decrement > low)
      throw new IllegalArgumentException();

    palette = new Color[size];

    for (int i = 0, r = high, g = 0, b = 0; i < palette.length ; i++)
      {
	palette[i] = new Color(r, g, b);

	// first play with red
	if (r >= low)
	  r -= decrement;
	else 
	  {
	    // then with green
	    r = 0;
	    g = high;
	  }

	if (g >= low)
	  g -= decrement;
	else 
	  {
	    // then with blue
	    g = 0;
	    b = high;
	  }

	if (b >= low)
	  b -= decrement;
	else 
	  {
	    // and then with red again
	    b = 0;
	    r = high;
	  }
      }
  }

  private void constructPalette(int size)
  {
    if (palette != null && size == palette.length)
      return;

    constructPalette(size, DEF_PAL_HIGH, DEF_PAL_LOW, DEF_PAL_DEC);
  }

  public FractalDrawPanel() 
  {
    addMouseMotionListener(this);
    addMouseListener(this);

    // we initialize a default fractal object that will be later
    // resized in paint(), so for now we just pass some bogus display
    // dimensions (small so that it won't have to compute much)
    constructPalette(FractalSettings.DEF_ITERATION_THRESHOLD);
    fractal = new Mandelbrot(1, 1, 
			     FractalSettings.DEF_X_MIN, 
			     FractalSettings.DEF_Y_MIN,
			     FractalSettings.DEF_X_MAX,
			     FractalSettings.DEF_Y_MAX, 
			     FractalSettings.DEF_ITERATION_THRESHOLD,
			     palette);
  }

  void drawFractal(String fractal_type, int iteration_threshold,
		   double x_min, double y_min,
		   double x_max, double y_max,
		   double julia_x, double julia_y)
  {
    Rectangle r = getBounds();

    constructPalette(iteration_threshold);

    if (fractal_type.equals(FractalSettings.MANDELBROT))
      fractal = new Mandelbrot(r.width, r.height, x_min, y_min, x_max, y_max,
			       iteration_threshold, palette);
    else if (fractal_type.equals(FractalSettings.JULIA))
      fractal = new Julia(r.width, r.height, x_min, y_min, x_max, y_max,
			  iteration_threshold, palette, julia_x ,julia_y);
    else if (fractal_type.equals(FractalSettings.FRACTAL110901))
      fractal = new Fractal110901(r.width, r.height, 
				  x_min, y_min, x_max, y_max,
				  iteration_threshold, palette);

    repaint();
  }

  void enhanceFractal(int iteration_threshold)
  {
    Rectangle r = getBounds();

    constructPalette(iteration_threshold);

    fractal.enhance(iteration_threshold, palette);

    repaint();
  }

  void setSettings(FractalSettings fs)
  {
    settings = fs;
  }

  public void mouseMoved(MouseEvent e) 
  {
    e.consume();
    
    // display fractal space coordinates
    Point p = new Point(e.getX(), e.getY());
    Point2D.Double pf = fractal.convertToFractalSpace(p);
    settings.displayCoordinates(pf.getX(), pf.getY());
  }

  // a smarter drawRect() method
  private void drawRect(Graphics g, int x1, int y1, int x2, int y2)
  {
    // arrange for point 1 to be upper-left point of region
    // and for point 2 to be the bottom-right point
    int tmp;
    if (x2 < x1)
      {
	tmp = x1;
	x1 = x2;
	x2 = tmp;
      }
    if (y2 < y1)
      {
	tmp = y1;
	y1 = y2;
	y2 = tmp;
      }

    g.drawRect(x1, y1, x2 - x1, y2 - y1);
  }
  
  public void mousePressed(MouseEvent e) 
  {
    e.consume();

    x1 = e.getX();
    y1 = e.getY();
    x2 = -1;
  }

  // this will draw the zoom rectangle
  public void mouseDragged(MouseEvent e) 
  {
    e.consume();

    bDragging = true;

    Graphics g = getGraphics();
    g.setXORMode(Color.white);

    // if this is not the second point then we have a rectangle to erase
    if (x2 != -1)
      drawRect(g, x1, y1, x2, y2);

    // draw a new zoom rectangle
    x2 = e.getX();
    y2 = e.getY();
    drawRect(g, x1, y1, x2, y2);
  }

  public void mouseReleased(MouseEvent e) 
  {
    e.consume();

    if (bDragging)
      {
	bDragging = false;

	Graphics g = getGraphics();

	// erase the zoom rectangle
	g.setXORMode(Color.white);
	drawRect(g, x1, y1, x2, y2);

	x2 = e.getX();
	y2 = e.getY();
	fractal.zoom(x1, y1, x2, y2);

	repaint();
      }
  }

  public void mouseEntered(MouseEvent e) 
  {
  }

  public void mouseExited(MouseEvent e) 
  {
  }

  public void mouseClicked(MouseEvent e) 
  {
    settings.unhide();
  }

  public void paint(Graphics g) 
  {
    Rectangle r = getBounds();

    // clean up background
    g.fillRect(0, 0, r.width, r.height);

    // resize fractal if screen dimensions have changed
    if (r.width != old_width || r.height != old_height)
      {
	try
	  {
	    fractal.resizeDisplay(r.width, r.height);
	  }
	catch (IllegalArgumentException invalid_dimensions)
	  {
	    // do nothing, it can happen when the user made the window
	    // so small that the display area disappeared
	  }

	old_width = r.width;
	old_height = r.height;
      }

    fractal.display(g);
  }
}

class FractalSettings 
  extends JPanel 
  implements ItemListener, ActionListener 
{
  public static final String MANDELBROT    = "Mandelbrot";
  public static final String JULIA         = "Julia";
  public static final String FRACTAL110901 = "Fractal110901";

  public static final int DEF_ITERATION_THRESHOLD = 100;
  public static final double DEF_X_MIN = -2;
  public static final double DEF_Y_MIN = -2;
  public static final double DEF_X_MAX = 2;
  public static final double DEF_Y_MAX = 2;
  public static final double DEF_JULIA_X = -0.5;
  public static final double DEF_JULIA_Y = 0.5;

  public static final int EDIT_SIZE = 5;

  public static final int DOUBLE_DECIMALS_DISPLAYED = 7; 

  private FractalDrawer parent;
  private FractalDrawPanel draw_panel;

  private String fractal_type;

  private JTextField txt_iteration_threshold;
  private JTextField txt_x_min;
  private JTextField txt_x_max;
  private JTextField txt_y_min;
  private JTextField txt_y_max;
  private JTextField txt_julia_x;
  private JTextField txt_julia_y;

  private JComboBox fractal_type_choice;

  private JLabel lbl_coord;

  private JButton btn_apply_all;
  private JButton btn_apply_threshold;
  private JButton btn_hide;

  private int iteration_threshold;
  private double x_min, y_min, x_max, y_max, julia_x, julia_y;

  public FractalSettings(FractalDrawer parent, FractalDrawPanel draw_panel) 
  {
    super(false);

    this.parent = parent;
    this.draw_panel = draw_panel;
    draw_panel.setSettings(this);

    setLayout(new BorderLayout());


    // Warning! Panelomania ahead!
    JPanel north = new JPanel();
    add("North", north);

    JPanel center = new JPanel();
    center.setLayout(new BorderLayout());
    add("Center", center);

    JPanel center_north = new JPanel();
    center.add("North", center_north);

    JPanel center_center = new JPanel();
    center.add("Center", center_center);

    JPanel center_south = new JPanel();
    center.add("South", center_south);

    JPanel south = new JPanel();
    south.setLayout(new BorderLayout());
    add("South", south);

    JPanel south_north = new JPanel();
    south.add("North", south_north);

    JPanel south_center = new JPanel();
    south.add("Center", south_center);

    JPanel south_south = new JPanel();
    south.add("South", south_south);


    // now adding controls to panels
    String[] fractal_types = {MANDELBROT, JULIA, FRACTAL110901};
    fractal_type_choice = new JComboBox(fractal_types);
    fractal_type_choice.setSelectedIndex(0);
    fractal_type_choice.addItemListener(this);
    north.add(fractal_type_choice);
    fractal_type = MANDELBROT;

    JLabel lbl_temp = new JLabel("Iteration threshold:");
    lbl_temp.setForeground(Color.black);
    north.add(lbl_temp);
    north.add(txt_iteration_threshold 
	      = new JTextField("" + DEF_ITERATION_THRESHOLD, EDIT_SIZE));

    lbl_temp = new JLabel("x_min:");
    lbl_temp.setForeground(Color.black);
    center_north.add(lbl_temp);
    center_north.add(txt_x_min
		     = new JTextField("" + DEF_X_MIN, EDIT_SIZE));
    lbl_temp = new JLabel("x_max:");
    lbl_temp.setForeground(Color.black);
    center_north.add(lbl_temp);
    center_north.add(txt_x_max
		     = new JTextField("" + DEF_X_MAX, EDIT_SIZE));

    lbl_temp = new JLabel("y_min:");
    lbl_temp.setForeground(Color.black);
    center_center.add(lbl_temp);
    center_center.add(txt_y_min
		      = new JTextField("" + DEF_Y_MIN, EDIT_SIZE));
    lbl_temp = new JLabel("y_max:");
    lbl_temp.setForeground(Color.black);
    center_center.add(lbl_temp);
    center_center.add(txt_y_max
		      = new JTextField("" + DEF_Y_MAX, EDIT_SIZE));

    lbl_temp = new JLabel("julia_x:");
    lbl_temp.setForeground(Color.black);
    center_south.add(lbl_temp);
    center_south.add(txt_julia_x
		     = new JTextField("" + DEF_JULIA_X, EDIT_SIZE));
    txt_julia_x.setEnabled(false);
    lbl_temp = new JLabel("julia_y:");
    lbl_temp.setForeground(Color.black);
    center_south.add(lbl_temp);
    center_south.add(txt_julia_y
		     = new JTextField("" + DEF_JULIA_Y, EDIT_SIZE));
    txt_julia_y.setEnabled(false);

    south_north.add(lbl_coord = new JLabel("(0.000000000, 0.000000000)", 
					   SwingConstants.CENTER));
    lbl_coord.setForeground(Color.black);

    south_center.add(btn_apply_all = new JButton("Apply all"));
    btn_apply_all.addActionListener(this);
    south_center.add(btn_apply_threshold = new JButton("Apply threshold"));
    btn_apply_threshold.addActionListener(this);
    south_center.add(btn_hide = new JButton("Hide settings"));
    btn_hide.addActionListener(this);

    lbl_temp = new JLabel(FractalDrawer.INFO);
    south_south.add(lbl_temp);
  }

  // if there are more than num_decimals in the String representation
  // of a double value then throw away extra decimals
  private String truncate(String s, int num_decimals)
  {
    return s.substring(0, Math.min(s.indexOf(".") + num_decimals + 1, 
				   s.length() - 1));
  }

  // display fractal space coordinates of mouse pointer
  void displayCoordinates(double x, double y)
  {
    String sx = "" + x;
    String sy = "" + y;
    lbl_coord.setText("(" + truncate(sx, DOUBLE_DECIMALS_DISPLAYED) 
		      + ", " + truncate(sy, DOUBLE_DECIMALS_DISPLAYED) + ")");
  }

  // unhide this panel, this method will be called from draw_panel as
  // a result of a mouse click
  void unhide()
  {
    if(! isVisible())
      {
	setVisible(true);

	// now make place for us by shrinking the draw_panel
	Rectangle parent_r = parent.getBounds();
	Rectangle my_r = getBounds();
	draw_panel.setSize(parent_r.width, parent_r.height - my_r.height);
      }
  }

  public void itemStateChanged(ItemEvent event) 
  {
    if (event.getSource() == fractal_type_choice) 
      {
	fractal_type = (String) event.getItem();
	if (fractal_type.equals(JULIA))
	  {
	    txt_julia_x.setEnabled(true);
	    txt_julia_y.setEnabled(true);
	  }
	else
	  {
	    txt_julia_x.setEnabled(false);
	    txt_julia_y.setEnabled(false);
	  }
      }
  }

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() == btn_apply_all) 
      {
	try
	  {
	    iteration_threshold = get_iteration_threshold();
	    x_min = get_x_min();
	    y_min = get_y_min();
	    x_max = get_x_max();
	    y_max = get_y_max();
	    julia_x = get_julia_x();
	    julia_y = get_julia_y();

	    draw_panel.drawFractal(fractal_type, iteration_threshold,
				   x_min, y_min, x_max, y_max,
				   julia_x, julia_y);
	  }
	catch (IllegalArgumentException invalid_input)
	  {
	    // wait for user to input something reasonable
	  }
      } 
    else if (event.getSource() == btn_apply_threshold) 
      {
	try
	  {
	    iteration_threshold = get_iteration_threshold();

	    draw_panel.enhanceFractal(iteration_threshold);
	  }
	catch (IllegalArgumentException invalid_input)
	  {
	    // wait for user to input something reasonable
	  }
      }
    else if (event.getSource() == btn_hide)
      {
	setVisible(false);

	// resize the draw_panel to use our display space
	Rectangle r = parent.getBounds();
	draw_panel.setSize(r.width, r.height);
      } 
  }

  private void dealWithInvalidValue(JTextField txt)
  {
    txt.setText("");
    txt.requestFocus();
    throw new IllegalArgumentException();
  }

  private int get_iteration_threshold()
  {
    String s = txt_iteration_threshold.getText();

    try
      {
	int value = Integer.parseInt(s);
	if (value > 0)
	  return value;
	else
	  dealWithInvalidValue(txt_iteration_threshold);
      }
    catch (NumberFormatException e)
      {
	dealWithInvalidValue(txt_iteration_threshold);
      }

    return 0;
  }

  private double get_x_min()
  {
    String s = txt_x_min.getText();

    try
      {
	double value = Double.parseDouble(s);
	return value;
      }
    catch (NumberFormatException e)
      {
	dealWithInvalidValue(txt_x_min);
      }

    return 0.0;
  }

  private double get_y_min()
  {
    String s = txt_y_min.getText();

    try
      {
	double value = Double.parseDouble(s);
	return value;
      }
    catch (NumberFormatException e)
      {
	dealWithInvalidValue(txt_y_min);
      }

    return 0.0;
  }

  private double get_x_max()
  {
    String s = txt_x_max.getText();

    try
      {
	double value = Double.parseDouble(s);
	if (value > x_min)
	  return value;
	else
	  dealWithInvalidValue(txt_x_max);
      }
    catch (NumberFormatException e)
      {
	dealWithInvalidValue(txt_x_max);
      }

    return 0.0;
  }

  private double get_y_max()
  {
    String s = txt_y_max.getText();

    try
      {
	double value = Double.parseDouble(s);
	if (value > y_min)
	  return value;
	else
	  dealWithInvalidValue(txt_y_max);
      }
    catch (NumberFormatException e)
      {
	dealWithInvalidValue(txt_y_max);
      }

    return 0.0;
  }

  private double get_julia_x()
  {
    if (!fractal_type.equals(JULIA))
      return 0.0;

    String s = txt_julia_x.getText();

    try
      {
	double value = Double.parseDouble(s);
	return value;
      }
    catch (NumberFormatException e)
      {
	dealWithInvalidValue(txt_julia_x);
      }

    return 0.0;
  }

  private double get_julia_y()
  {
    if (!fractal_type.equals(JULIA))
      return 0.0;

    String s = txt_julia_y.getText();

    try
      {
	double value = Double.parseDouble(s);
	return value;
      }
    catch (NumberFormatException e)
      {
	dealWithInvalidValue(txt_julia_y);
      }

    return 0.0;
  }
}
