/*

FractalDrawer - A Fractal Drawer Applet/Application
Copyright (C) 2001  Laurentiu Cristofor


This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
USA

*/

/*
 * Mandelbrot.java       v1.0   11/09/2001
 *
 * the Mandelbrot fractal
 *
 * Copyright (c) 2001 Laurentiu Cristofor. All Rights Reserved.
 *
 */

import java.awt.Color;

public class Mandelbrot extends Fractal
{
  /**
   * Initializes the Mandelbrot fractal.
   *
   * @param display_width   the width of the display 
   * @param display_height   the height of the display
   * @param x_min   lowest real value of fractal space that we explore
   * @param y_min   lowest imaginary value of fractal space
   * @param x_max   largest real value of fractal space
   * @param y_max   largest imaginary value of fractal space
   * @param iteration_threshold   how many iterations we perform until 
   * we accept a point (unless we reject it after fewer iterations)
   * @param palette   an array of colors used to draw the fractal
   *
   * @exception InvalidArgumentException thrown if display width or
   * height are negative, or if x_min > x_max, or if y_min > y_max, or
   * if palette is null, or if iteration_threshold is <= 0, or if
   * palette has size less than iteration_threshold.
   **/
  public Mandelbrot(int display_width, int display_height,
		    double x_min, double y_min, double x_max, double y_max,
		    int iteration_threshold,
		    Color[] palette)
  {
    super(display_width, display_height,
	  x_min, y_min, x_max, y_max,
	  iteration_threshold, palette);

    TYPE_SYMMETRY = HORIZONTAL_SYMMETRY;
  }

  /**
   * Computes whether a point belongs to the fractal space.
   * (See if Zn+1 = Zn*Zn + C converges, with Z0 = 0,
   *  and C=(x + i*y))
   *
   * @param x   the real value of the point
   * @param y   the complex value of the point
   **/
  public int computeFractalValue(double x, double y)
  {
    double tmp, z_x = x, z_y = y;
    int iteration = 1;

    while (z_x*z_x + z_y*z_y < DISTANCE_TO_ORIGIN_SQUARED
	   && iteration < iteration_threshold)
      {
	tmp = z_x*z_x - z_y*z_y + x; 
	z_y = 2*z_x*z_y + y; 
	z_x = tmp;
	iteration++;
      }

    return iteration;
  }
}
